<?php

class sfCacheManager
{
    private static $cache = null;
    private static $tagKeyPrefix = 'TAGS_INFO_';

    /**
     * Initializes this cache instance.
     */
    private static function getInstance ()
    {
        if (is_null(self::$cache) && sfConfig::get('sf_cache')) {
            $options = sfConfig::get('sf_cache_settings_param');
            $class = sfConfig::get('sf_cache_settings_class');

            if ('sfAPCCache' == $class && function_exists('apc_store')) {
                self::$cache = new sfAPCCache($options);
            } elseif ('sfXCacheCache' == $class && function_exists('xcache_set')) {
                self::$cache = new sfXCacheCache($options);
            } elseif ('sfEAcceleratorCache' == $class && function_exists('eaccelerator_put')) {
                self::$cache = new sfEAcceleratorCache($options);
            } elseif ('sfSQLiteCache' == $class && extension_loaded('SQLite')) {
                self::$cache = new sfSQLiteCache($options);
            } elseif ('sfMemcacheCache' == $class && class_exists('Memcache')) {
                self::$cache = new sfMemcacheCache($options);
            } else {
                self::$cache = new sfFileCache($options);
            }

        }

        return self::$cache;
    }

    public static function getObjectTag($object)
    {
        return get_class($object) . '_' . $object->getId();
    }

    /**
     * Возвращает объект
     *
     * @param unknown_type $class
     * @param unknown_type $id
     * @return unknown
     */
    public static function getObject($class, $id, $lifetime = null)
    {
        if (!$id)
            return null;

        $key = 'OBJECT_' . $class . '_' . $id;

        if (!$object = self::get($key))  {

            $object = call_user_func(array($class . 'Peer', 'retrieveByPk'), $id);

            if ($object) {

                if (!is_null($lifetime))
                    self::set($key, $object, array($class, self::getObjectTag($object)), $lifetime);
                else
                    self::set($key, $object, array($class, self::getObjectTag($object)));

            }

        }

        return $object;
    }

    public static function getImageSize($filename)
    {
        $fileInfo = @stat($filename);

        if (!$fileInfo)
            return null;

        $key = 'IMAGE_SIZE_' . md5($filename . $fileInfo['size'] . $fileInfo['mtime']);

        if (self::has($key))

            return self::get($key);

        else {

            $width = 0; $height = 0;
            list($width, $height) = @getimagesize($filename);

            if ($width && $height) {

                $result = array($width, $height);

                self::set($key, $result);

            }

        }

        return isset($result) ? $result : null;
    }

    /**
     * @see sfCache
     */
    public static function get ($key, $default = null)
    {
        if (!self::getInstance())
            return null;

        $data = self::getInstance()->get($key);

        if ($data === null) {
            return $default;
        }

        $data = unserialize($data);

        if (is_array($data) && (isset($data['Tags']))) { // Если запись помечена тэгами

            if (self::checkTags($data['Tags'])) { // Теги актуальные

                return $data['Data'];

            } else { // Существуют устаревшие теги

                self::getInstance()->remove($key);

                return null;

            }

        } else {

            return $data;

        }
    }

    /**
     * @see sfCache
     */
    public static function has ($key)
    {
        return !is_null(self::get($key));
    }

    /**
     * @see sfCache
     */
    public static function set ($key, $value, $tags = array(), $lifetime = null)
    {
        if (!self::getInstance())
            return null;

        if (!is_array($tags))
            throw new sfException('sfCacheManager: The list of tags should be an array.');

        if (count($tags) > 0) {

            $tagsData = array();

            foreach ($tags as $tag) {
                $tagsData[$tag] = self::getTagVersion($tag);
            }

            $value = array('Tags' => $tagsData, 'Data' => $value);
        }

        $data = serialize($value);

        return self::getInstance()->set($key, $data, $lifetime);
    }

    /**
     * @see sfCache
     */
    public static function remove ($key)
    {
        if (!self::getInstance())
            return null;

        return self::getInstance()->remove($key);
    }

    /**
     * @see sfCache
     */
    public static function removePattern ($pattern)
    {
        if (!self::getInstance())
            return null;

        self::getInstance()->removePattern($pattern);
    }

    public static function removeObject ($class, $id)
    {
        if (!self::getInstance())
            return null;

        $object = self::getObject($class, $id);

        if ($object)
            self::removeTag(self::getObjectTag($object));
    }

    public static function removeTag ($tag)
    {
        $key = self::getTagCacheKey($tag);

        return self::getInstance()->remove($key);
    }

    /**
     * @see sfCache
     */
    public function clean ($mode = sfCache::ALL)
    {
        if (!self::getInstance())
            return null;

        return self::getInstance()->removePattern($mode);
    }

    /**
     * @see sfCache
     */
    public function getTimeout ($key)
    {
        if (!self::getInstance())
            return null;

        return self::getInstance()->getTimeout($key);
    }

    /**
     * @see sfCache
     */
    public function getLastModified ($key)
    {
        if (!self::getInstance())
            return null;

        return self::getInstance()->getLastModified($key);
    }

    public static function checkTags ($tags)
    {
        foreach ($tags as $tag => $version) {

            $versionCurrent = self::getTagVersion($tag);

            if ($versionCurrent != $version)
                return false;

        }

        return true;
    }

    public static function getTagVersion ($tag)
    {
        $key = self::getTagCacheKey($tag);
        if (!$version = self::getInstance()->get($key)) {
            $version = microtime(true);
            self::getInstance()->set($key, $version);
        }

        return $version;
    }

    public static function getTagCacheKey ($tag)
    {
        return self::$tagKeyPrefix . $tag;
    }
}