<?php

class Lot extends BaseLot
{
    private $_monthName = array('января', 'февраля', 'марта', 'апреля', 'мая', 'июня', 'июля', 'августа', 'сентября', 'октября', 'ноября', 'декабря');

    public function __toString()
    {
        return '#' . $this->getId() . ': ' . $this->getProductTitle();
    }

    public function getProductTitle()
    {
        return ($product = $this->getProduct()) ? $product->getTitle() : '-';
    }

    public function getPriceFormatted()
    {
        return number_format($this->getPrice(), 0, ',', ' ');
    }

    public function getTimeRemaining()
    {
        return strtotime($this->getDtEnd()) - time();
    }

    public function getTimeRemainingFormatted()
    {
        $timestampRemaining = $this->getTimeRemaining();

//        $days = floor($timestampRemaining / 86400);
        $days = 0;
        $hours = floor(($timestampRemaining - $days * 86400) / 3600);
        $minutes = floor(($timestampRemaining - $days * 86400 - $hours * 3600) / 60);
        $seconds = $timestampRemaining - $days * 86400 - $hours * 3600 - $minutes * 60;

        return $days ?
            sprintf('%d:%02d:%02d:%02d', $days, $hours, $minutes, $seconds) :
            sprintf('%d:%02d:%02d', $hours, $minutes, $seconds);
    }

    public function getProfitPercentage()
    {
        $product = sfCacheManager::getObject('Product', $this->getProductId());

        return 100 - ceil($this->getPrice() / $product->getPrice() * 100);
    }

    public function getWinnerProfitPercentage()
    {
        $product = sfCacheManager::getObject('Product', $this->getProductId());

        $result = 100 - ceil(($this->getPrice() + $this->getUserBetsPrice()) / $product->getPrice() * 100);

        return max(0, $result);
    }

    public function getUserBetsCount($userId = null)
    {
        if (is_null($userId))
            $userId = $this->getBet()->getUserId();

        $cacheKey = 'LOT_' . $this->getId() . '_WINNER_BETS_COUNT_' . $userId;

        if (sfCacheManager::has($cacheKey))
            return sfCacheManager::get($cacheKey);

        if (!$this->getBet() || !$userId) {

            $winnerBetsCount = 0;

        } else {

            $c = new Criteria();
            $c->add(BetPeer::LOT_ID, $this->getId());
            $c->add(BetPeer::USER_ID, $userId);

            $winnerBetsCount = BetPeer::doCount($c);
        }

        sfCacheManager::set($cacheKey, $winnerBetsCount, array(sfCacheManager::getObjectTag($this)), 30*24*3600);

        return $winnerBetsCount;
    }

    public function getUserBetsPrice($userId = null)
    {
        return $this->getUserBetsCount($userId) * sfConfig::get('app_lot_price_bet_default');
    }

    public function getWinnerBetsPriceFormatted()
    {
        return number_format($this->getUserBetsPrice(), 0, ',', ' ');
    }


    public function getUserBetsPriceFormatted($userId)
    {
        return number_format($this->getUserBetsPrice($userId), 0, ',', ' ');
    }

    public function getDtEndFormatted()
    {
        return sprintf(
            '%s %s в %s:%s',
            $this->getDtEnd('j'),
            $this->_monthName[$this->getDtEnd('n')-1],
            $this->getDtEnd('G'),
            $this->getDtEnd('i')
        );
    }

    public function getDtStartFormatted()
    {
        return sprintf(
            '%s %s в %s:%s',
            $this->getDtStart('j'),
            $this->_monthName[$this->getDtStart('n')-1],
            $this->getDtStart('G'),
            $this->getDtStart('i')
        );
    }

    public function getDtStartShortFormatted()
    {
        return sprintf(
            '%s %s',
            $this->getDtStart('j'),
            $this->_monthName[$this->getDtStart('n')-1]
        );
    }

    public function isOptionManual()
    {
        $cacheKey = 'LOT_' . $this->getId() . '_OPTION_IS_MANUAL';
        if (!sfCacheManager::has($cacheKey)) {

            $c = new Criteria();
            $c->addJoin(LotPeer::ID, LotAuctionOptionPeer::LOT_ID, Criteria::INNER_JOIN);
            $c->add(LotAuctionOptionPeer::AUCTION_OPTION_ID, sfConfig::get('app_lot_auction_option_manual'));
            $c->add(LotPeer::ID, $this->getId());

            $isManual = !!LotPeer::doSelectOne($c);

            sfCacheManager::set($cacheKey, $isManual, array('LotAuctionOptionList'), 30*24*3600);

        } else

            $isManual = sfCacheManager::get($cacheKey);

        return $isManual;

    }

    public function isOptionNovice()
    {
        $cacheKey = 'LOT_' . $this->getId() . '_OPTION_IS_NOVICE';
        if (!sfCacheManager::has($cacheKey)) {

            $c = new Criteria();
            $c->addJoin(LotPeer::ID, LotAuctionOptionPeer::LOT_ID, Criteria::INNER_JOIN);
            $c->add(LotAuctionOptionPeer::AUCTION_OPTION_ID, sfConfig::get('app_lot_auction_option_novice'));
            $c->add(LotPeer::ID, $this->getId());

            $isNovice = !!LotPeer::doSelectOne($c);

            sfCacheManager::set($cacheKey, $isNovice, array('LotAuctionOptionList'), 30*24*3600);

        } else

            $isNovice = sfCacheManager::get($cacheKey);

        return $isNovice;

    }

    public function isOptionCompensation()
    {
        $cacheKey = 'LOT_' . $this->getId() . '_OPTION_IS_COMPENSATION';
        if (!sfCacheManager::has($cacheKey)) {

            $c = new Criteria();
            $c->addJoin(LotPeer::ID, LotAuctionOptionPeer::LOT_ID, Criteria::INNER_JOIN);
            $c->add(LotAuctionOptionPeer::AUCTION_OPTION_ID, sfConfig::get('app_lot_auction_option_compensation'));
            $c->add(LotPeer::ID, $this->getId());

            $isReturn = !!LotPeer::doSelectOne($c);

            sfCacheManager::set($cacheKey, $isReturn, array('LotAuctionOptionList'), 30*24*3600);

        } else

            $isReturn = sfCacheManager::get($cacheKey);

        return $isReturn;

    }

    public function isClosedByTime()
    {
        return (0 > $this->getTimeRemaining()) || $this->getIsClosed();
    }

	public function setDtEnd ($v)
    {
        parent::setDtEnd($v);

        $this->setDtAutobet(strtotime($this->getDtEnd()) - sfConfig::get('app_lot_time_autobet_min', 3) - rand(0, 7));
    }

    public function save(PropelPDO $con = null)
    {
        if (!$this->getDtEnd())
            $this->setDtEnd($this->getDtStart());

        if (!$this->getPrice())
            $this->setPrice(sfConfig::get('app_lot_price_default'));

        if (!$this->getDtAutobet())
            $this->setDtAutobet(strtotime($this->getDtEnd()) - sfConfig::get('app_lot_time_autobet_min', 3) - rand(0, 10-sfConfig::get('app_lot_time_autobet_min', 3)));

        parent::save($con);

        $this->clearCache();
    }

    public function delete(PropelPDO $con = null)
    {
        $this->clearCache();

        parent::delete($con);
    }

    private function clearCache()
    {
        // Сам объект
        sfCacheManager::removeObject('Lot', $this->getId());

        // Все списки
        sfCacheManager::removeTag('LotList');

        // История ставок
        sfCacheManager::remove('BETS_HISTORY_LOT_' . $this->getId());

        // Опции
        sfCacheManager::remove('LOT_' . $this->getId() . '_OPTION_IS_MANUAL');
        sfCacheManager::remove('LOT_' . $this->getId() . '_OPTION_IS_NOVICE');
        sfCacheManager::remove('LOT_' . $this->getId() . '_OPTION_IS_RETURN');

        // Обновление статических данных
        sfStaticManager::buildLotsStatus();
    }
}
